/*
 * Decompiled with CFR 0.152.
 */
package org.evosuite.setup;

import com.thoughtworks.xstream.XStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.evosuite.Properties;
import org.evosuite.classpath.ResourceList;
import org.evosuite.rmi.ClientServices;
import org.evosuite.setup.InheritanceTree;
import org.evosuite.statistics.RuntimeVariable;
import org.evosuite.utils.LoggingUtils;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.tree.ClassNode;
import org.objectweb.asm.tree.InnerClassNode;
import org.objectweb.asm.tree.MethodNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InheritanceTreeGenerator {
    private static Logger logger = LoggerFactory.getLogger(InheritanceTreeGenerator.class);
    private static final Pattern ANONYMOUS_MATCHER1 = Pattern.compile(".*\\$\\d+.*$");
    private static final Pattern ANONYMOUS_MATCHER2 = Pattern.compile(".*\\.\\d+.*$");
    private static List<String> classExceptions = Arrays.asList("java/lang/Class", "java/lang/Object", "java/lang/String", "java/lang/Comparable", "java/io/Serializable", "com/apple", "apple/", "sun/", "com/sun", "com/oracle", "sun/awt", "java/util/prefs/MacOSXPreferences");

    public static InheritanceTree createFromClassPath(List<String> classPath) {
        if (!Properties.INSTRUMENT_CONTEXT && !Properties.INHERITANCE_FILE.isEmpty()) {
            try {
                InheritanceTree tree = InheritanceTreeGenerator.readInheritanceTree(Properties.INHERITANCE_FILE);
                LoggingUtils.getEvoLogger().info("* Inheritance tree loaded from {}", (Object)Properties.INHERITANCE_FILE);
                return tree;
            }
            catch (IOException e) {
                LoggingUtils.getEvoLogger().warn("* Error loading inheritance tree: {}", (Throwable)e);
            }
        }
        logger.debug("Reading JDK data");
        InheritanceTree inheritanceTree = InheritanceTreeGenerator.readJDKData();
        if (inheritanceTree == null) {
            inheritanceTree = new InheritanceTree();
        }
        logger.debug("CP: " + classPath);
        for (String classPathEntry : classPath) {
            logger.debug("Looking at CP entry: " + classPathEntry);
            if (classPathEntry.isEmpty() || classPathEntry.matches(".*evosuite-.*\\.jar")) continue;
            logger.debug("Analyzing classpath entry " + classPathEntry);
            LoggingUtils.getEvoLogger().info("  - " + classPathEntry);
            InheritanceTreeGenerator.analyze(inheritanceTree, classPathEntry);
        }
        return inheritanceTree;
    }

    public static InheritanceTree createFromClassList(Collection<String> classNames) throws IllegalArgumentException {
        if (classNames == null || classNames.isEmpty()) {
            throw new IllegalArgumentException("No class name defined");
        }
        InheritanceTree inheritanceTree = new InheritanceTree();
        for (String className : classNames) {
            if (className == null) {
                throw new IllegalArgumentException("Null class name");
            }
            InheritanceTreeGenerator.analyzeClassName(inheritanceTree, className);
        }
        HashSet<String> classes = new HashSet<String>(inheritanceTree.getAllClasses());
        for (String className : classes) {
            if (classNames.contains(className)) continue;
            inheritanceTree.removeClass(className);
        }
        return inheritanceTree;
    }

    public static void gatherStatistics(InheritanceTree inheritanceTree) {
        ClientServices.getInstance().getClientNode().trackOutputVariable(RuntimeVariable.Classpath_Classes, inheritanceTree.getNumClasses());
    }

    private static void analyze(InheritanceTree inheritanceTree, String entry) {
        InheritanceTreeGenerator.analyze(inheritanceTree, new File(entry));
    }

    private static void analyze(InheritanceTree inheritanceTree, File file) {
        if (!file.canRead()) {
            return;
        }
        if (file.getName().endsWith(".jar")) {
            InheritanceTreeGenerator.analyzeJarFile(inheritanceTree, file);
        } else if (file.getName().endsWith(".class")) {
            InheritanceTreeGenerator.analyzeClassFile(inheritanceTree, file);
        } else if (file.isDirectory()) {
            InheritanceTreeGenerator.analyzeDirectory(inheritanceTree, file);
        }
    }

    private static void analyzeJarFile(InheritanceTree inheritanceTree, File jarFile) {
        ZipFile zf;
        try {
            zf = new ZipFile(jarFile);
        }
        catch (Exception e) {
            logger.warn("Failed to open/analyze jar file " + jarFile.getAbsolutePath() + " , " + e.getMessage());
            return;
        }
        Enumeration<? extends ZipEntry> e = zf.entries();
        while (e.hasMoreElements()) {
            ZipEntry ze = e.nextElement();
            String fileName = ze.getName();
            if (!fileName.endsWith(".class")) continue;
            try {
                InheritanceTreeGenerator.analyzeClassStream(inheritanceTree, zf.getInputStream(ze), false);
            }
            catch (IOException e1) {
                logger.error("Error while analyzing class " + fileName + " in the jar " + jarFile.getAbsolutePath(), (Throwable)e1);
            }
        }
        try {
            zf.close();
        }
        catch (IOException e1) {
            logger.warn("Failed to close jar file " + jarFile.getAbsolutePath() + " , " + e1.getMessage());
        }
    }

    private static void analyzeDirectory(InheritanceTree inheritanceTree, File directory) {
        for (File file : directory.listFiles()) {
            InheritanceTreeGenerator.analyze(inheritanceTree, file);
        }
    }

    private static void analyzeClassFile(InheritanceTree inheritanceTree, File classFile) {
        try {
            InheritanceTreeGenerator.analyzeClassStream(inheritanceTree, new FileInputStream(classFile), false);
        }
        catch (FileNotFoundException e) {
            logger.error("", (Throwable)e);
        }
    }

    private static void analyzeClassName(InheritanceTree inheritanceTree, String className) {
        InputStream stream = ResourceList.getClassAsStream(className);
        if (stream == null) {
            throw new IllegalArgumentException("Failed to locate/load class: " + className);
        }
        logger.debug("Going to analyze: " + className);
        InheritanceTreeGenerator.analyzeClassStream(inheritanceTree, stream, false);
    }

    private static void analyzeClassStream(InheritanceTree inheritanceTree, InputStream inputStream, boolean onlyPublic) {
        try {
            ClassReader reader = new ClassReader(inputStream);
            inputStream.close();
            ClassNode cn = new ClassNode();
            reader.accept((ClassVisitor)cn, 7);
            InheritanceTreeGenerator.analyzeClassNode(inheritanceTree, cn, onlyPublic);
        }
        catch (IOException e) {
            logger.error("", (Throwable)e);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            logger.error("ASM Error while reading class (" + e.getMessage() + ")");
        }
    }

    private static void analyzeClassNode(InheritanceTree inheritanceTree, ClassNode cn, boolean onlyPublic) {
        logger.info("Analyzing class " + cn.name);
        if ((0x200 & cn.access) != 512) {
            for (Object m : cn.methods) {
                MethodNode mn = (MethodNode)m;
                inheritanceTree.addAnalyzedMethod(cn.name, mn.name, mn.desc);
            }
            if ((0x400 & cn.access) == 1024) {
                inheritanceTree.registerAbstractClass(cn.name);
            }
        } else {
            inheritanceTree.registerInterface(cn.name);
        }
        if (onlyPublic) {
            if ((cn.access & 1) == 0) {
                return;
            }
        } else if (!InheritanceTreeGenerator.canUse(cn)) {
            logger.info("Cannot use " + cn.name);
            return;
        }
        if (cn.superName != null) {
            inheritanceTree.addSuperclass(cn.name, cn.superName, cn.access);
        }
        List interfaces = cn.interfaces;
        for (String interfaceName : interfaces) {
            inheritanceTree.addInterface(cn.name, interfaceName);
        }
    }

    public static boolean canUse(ClassNode cn) {
        if ((cn.access & 2) == 2) {
            logger.debug(cn.name + " is private, ignoring it");
            return false;
        }
        if (ANONYMOUS_MATCHER1.matcher(cn.name).matches()) {
            logger.debug(cn.name + " looks like an anonymous class, ignoring it");
            return false;
        }
        if (ANONYMOUS_MATCHER2.matcher(cn.name).matches()) {
            logger.debug(cn.name + " looks like an anonymous class, ignoring it");
            return false;
        }
        if (cn.name.startsWith("junit")) {
            return false;
        }
        List in = cn.innerClasses;
        for (InnerClassNode inc : in) {
            if (!cn.name.equals(inc.name)) continue;
            if ((inc.access & 2) == 2) {
                return false;
            }
            logger.debug("Can use inner class: " + inc.name);
            return true;
        }
        logger.debug(cn.name + " is accessible");
        return true;
    }

    public static void generateJDKCluster(String ... filters) {
        int counter = 0;
        Collection<String> list = InheritanceTreeGenerator.getAllResources();
        InheritanceTree inheritanceTree = new InheritanceTree();
        ArrayList<InheritanceTree> others = new ArrayList<InheritanceTree>();
        for (String filterFile : filters) {
            logger.info("Trying to load " + filterFile);
            try {
                InheritanceTree tree = InheritanceTreeGenerator.readUncompressedInheritanceTree(filterFile);
                others.add(tree);
            }
            catch (IOException e) {
                logger.info("Error: " + e);
            }
        }
        block5: for (String name : list) {
            for (String exception : classExceptions) {
                if (!name.startsWith(exception.replace('/', '.'))) continue;
                logger.info("Skipping excluded class " + name);
                continue block5;
            }
            for (InheritanceTree other : others) {
                if (!other.hasClass(ResourceList.getClassNameFromResourcePath(name))) {
                    logger.info("Skipping " + name + " because it is not in other inheritance tree");
                    continue block5;
                }
                logger.info("Not skipping " + name + " because it is in other inheritance tree");
            }
            if (name.matches(".*\\$\\d+$")) {
                logger.info("Skipping anonymous class");
                continue;
            }
            InputStream stream = ResourceList.getClassAsStream(name);
            if (stream == null) {
                logger.warn("Cannot open/find " + name);
                continue;
            }
            InheritanceTreeGenerator.analyzeClassStream(inheritanceTree, stream, true);
            ++counter;
        }
        logger.info("Finished checking classes, writing data for " + counter + " classes");
        try {
            FileOutputStream stream = new FileOutputStream(new File("client/src/main/resources/JDK_inheritance.xml"));
            XStream xstream = new XStream();
            xstream.toXML((Object)inheritanceTree, (OutputStream)stream);
        }
        catch (FileNotFoundException e) {
            logger.error("", (Throwable)e);
        }
    }

    public static InheritanceTree readJDKData() {
        XStream xstream = new XStream();
        String fileName = "/JDK_inheritance.xml";
        InputStream inheritance = InheritanceTreeGenerator.class.getResourceAsStream(fileName);
        if (inheritance != null) {
            return (InheritanceTree)xstream.fromXML(inheritance);
        }
        logger.warn("Found no JDK inheritance tree in the resource path: " + fileName);
        return null;
    }

    public static InheritanceTree readInheritanceTree(String fileName) throws IOException {
        XStream xstream = new XStream();
        GZIPInputStream inheritance = new GZIPInputStream(new FileInputStream(new File(fileName)));
        return (InheritanceTree)xstream.fromXML((InputStream)inheritance);
    }

    public static InheritanceTree readUncompressedInheritanceTree(String fileName) throws IOException {
        XStream xstream = new XStream();
        FileInputStream inheritance = new FileInputStream(new File(fileName));
        return (InheritanceTree)xstream.fromXML((InputStream)inheritance);
    }

    public static void writeInheritanceTree(InheritanceTree tree, File file) throws IOException {
        XStream xstream = new XStream();
        GZIPOutputStream output = new GZIPOutputStream(new FileOutputStream(file));
        xstream.toXML((Object)tree, (OutputStream)output);
        output.close();
    }

    public static Collection<String> getAllResources() {
        Collection<String> retval = InheritanceTreeGenerator.getResources(System.getProperty("java.class.path", "."));
        retval.addAll(InheritanceTreeGenerator.getResources(System.getProperty("sun.boot.class.path")));
        return retval;
    }

    private static Collection<String> getResources(String classPath) {
        String[] classPathElements;
        ArrayList<String> retval = new ArrayList<String>();
        for (String element : classPathElements = classPath.split(File.pathSeparator)) {
            if (element.contains("evosuite")) continue;
            try {
                retval.addAll(ResourceList.getAllClasses(element, "", true, true));
            }
            catch (IllegalArgumentException e) {
                System.err.println("Does not exist: " + element);
            }
        }
        return retval;
    }

    public static void main(String[] args) {
        InheritanceTreeGenerator.generateJDKCluster(args);
    }
}

